<?php
namespace App\Http\Controllers;

use App\Models\Item;
use App\Models\InventoryTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ItemWebController extends Controller
{
    public function index(Request $request) {
        $query = Item::query();

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%");
            });
        }

        $items = $query->withCount('transactions')->orderBy('name')->paginate(20);

        $stats = [
            'total' => Item::count(),
            'with_value' => Item::whereNotNull('estimated_value')->where('estimated_value', '>', 0)->count(),
        ];

        return view('items.index', compact('items', 'stats'));
    }
    public function create() { return view('items.create'); }
    public function store(Request $request)
    { $data = $request->validate(['sku' => 'nullable|string','name' => 'required|string','unit' => 'nullable|string','estimated_value' => 'nullable|numeric']); Item::create($data); return redirect()->route('items.index'); }
    
    public function show(Item $item) {
        // Stock by Warehouse
        $stock_by_warehouse = InventoryTransaction::where('item_id', $item->id)
            ->select('warehouse_id', DB::raw("SUM(CASE WHEN type = 'in' THEN quantity WHEN type = 'out' THEN -quantity ELSE 0 END) as current_stock"))
            ->groupBy('warehouse_id')
            ->having('current_stock', '>', 0)
            ->with('warehouse')
            ->get();

        // Recent Transactions
        $recent_transactions = $item->transactions()
            ->with(['warehouse'])
            ->latest()
            ->take(10)
            ->get();
            
        return view('items.show', compact('item', 'stock_by_warehouse', 'recent_transactions'));
    }

    public function edit(Item $item) { return view('items.edit', compact('item')); }
    public function update(Request $request, Item $item)
    { $data = $request->validate(['sku' => 'nullable|string','name' => 'sometimes|string','unit' => 'nullable|string','estimated_value' => 'nullable|numeric']); $item->update($data); return redirect()->route('items.show',$item); }
    public function destroy(Item $item) { $item->delete(); return redirect()->route('items.index'); }
}
