<?php
namespace App\Http\Controllers;

use App\Models\Expense;
use App\Models\Beneficiary;
use App\Models\Project;
use App\Models\Campaign;
use App\Models\Workspace;
use App\Models\GuestHouse;
use Illuminate\Http\Request;

class ExpenseWebController extends Controller
{
    public function index(Request $request)
    {
        $projectId = $request->get('project_id');
        $campaignId = $request->get('campaign_id');
        $workspaceId = $request->get('workspace_id');
        $month = $request->get('month');
        $year = $request->get('year');

        $expenses = Expense::with(['beneficiary', 'project', 'campaign', 'workspace'])
            ->when($projectId, function ($q) use ($projectId) {
                $q->where('project_id', $projectId);
            })
            ->when($campaignId, function ($q) use ($campaignId) {
                $q->where('campaign_id', $campaignId);
            })
            ->when($workspaceId, function ($q) use ($workspaceId) {
                $q->where('workspace_id', $workspaceId);
            })
            ->when($month, function ($q) use ($month) {
                $q->whereMonth('created_at', $month);
            })
            ->when($year, function ($q) use ($year) {
                $q->whereYear('created_at', $year);
            })
            ->orderByDesc('id')
            ->paginate(20);
        return view('expenses.index', compact('expenses'));
    }

    public function export(Request $request)
    {
        $projectId = $request->get('project_id');
        $campaignId = $request->get('campaign_id');
        $workspaceId = $request->get('workspace_id');
        $month = $request->get('month');
        $year = $request->get('year');

        $query = Expense::with(['beneficiary', 'project', 'campaign', 'workspace'])
            ->when($projectId, function ($q) use ($projectId) {
                $q->where('project_id', $projectId);
            })
            ->when($campaignId, function ($q) use ($campaignId) {
                $q->where('campaign_id', $campaignId);
            })
            ->when($workspaceId, function ($q) use ($workspaceId) {
                $q->where('workspace_id', $workspaceId);
            })
            ->when($month, function ($q) use ($month) {
                $q->whereMonth('created_at', $month);
            })
            ->when($year, function ($q) use ($year) {
                $q->whereYear('created_at', $year);
            })
            ->orderByDesc('id');

        $filename = "expenses_" . date('Ymd_His') . ".csv";

        return response()->streamDownload(function () use ($query) {
            $handle = fopen('php://output', 'w');
            fprintf($handle, chr(0xEF) . chr(0xBB) . chr(0xBF)); // BOM for Excel

            fputcsv($handle, ['ID', 'الوصف', 'المبلغ', 'العملة', 'التصنيف', 'النوع', 'المشروع', 'الحملة', 'المسكن', 'التاريخ']);
            $query->chunk(100, function ($expenses) use ($handle) {
                foreach ($expenses as $e) {
                    fputcsv($handle, [
                        $e->id,
                        $e->description ?? '—',
                        $e->amount,
                        $e->currency,
                        $e->category ?? '—',
                        $e->type,
                        $e->project->name ?? '—',
                        $e->campaign->name ?? '—',
                        $e->workspace->name ?? '—',
                        $e->created_at->format('Y-m-d')
                    ]);
                }
            });
            fclose($handle);
        }, $filename, ['Content-Type' => 'text/csv']);
    }

    public function create()
    {
        $beneficiaries = Beneficiary::orderBy('full_name')->get();
        $projects = Project::query()
            ->where(function ($q) {
                $q->where('name', 'like', '%بعثاء%')
                    ->orWhere('name', 'like', '%زاد%')
                    ->orWhere('name', 'like', '%مدرار%')
                    ->orWhere('name', 'like', '%كسو%');
            })
            ->orderBy('name')
            ->get();
        $campaigns = Campaign::orderByDesc('season_year')->orderBy('name')->get();
        $workspaces = Workspace::where('status', '!=', 'maintenance')->orderBy('name')->get();
        $guestHouses = GuestHouse::orderBy('name')->get();

        return view('expenses.create', compact('beneficiaries', 'projects', 'campaigns', 'workspaces', 'guestHouses'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'type' => 'required|in:operational,aid,logistics',
            'category' => 'nullable|string|max:100',
            'amount' => 'required|numeric',
            'currency' => 'nullable|string',
            'payment_method' => 'nullable|string',
            'description' => 'nullable|string',
            'project_id' => 'nullable|exists:projects,id',
            'campaign_id' => 'nullable|exists:campaigns,id',
            'workspace_id' => 'nullable|exists:workspaces,id',
            'guest_house_id' => 'nullable|exists:guest_houses,id',
            'beneficiary_id' => 'nullable|exists:beneficiaries,id',
            'paid_at' => 'nullable|date'
        ]);

        // Ensure only one association is selected (optional logic, but good for data integrity)
        // For now, we allow multiple or mix, but typically an expense belongs to one entity.
        // Let's just save what is passed.

        $exp = Expense::create($data);
        if (class_exists(\App\Services\ExpenseService::class)) {
            \App\Services\ExpenseService::postCreate($exp);
        }
        return redirect()->route('expenses.show', $exp);
    }

    public function show(Expense $expense)
    {
        return view('expenses.show', compact('expense'));
    }

    public function edit(Expense $expense)
    {
        $beneficiaries = Beneficiary::orderBy('full_name')->get();
        $projects = Project::query()
            ->where(function ($q) {
                $q->where('name', 'like', '%بعثاء%')
                    ->orWhere('name', 'like', '%زاد%')
                    ->orWhere('name', 'like', '%مدرار%')
                    ->orWhere('name', 'like', '%كسو%');
            })
            ->orderBy('name')
            ->get();
        $campaigns = Campaign::orderByDesc('season_year')->orderBy('name')->get();
        $workspaces = Workspace::where('status', '!=', 'maintenance')->orderBy('name')->get();
        $guestHouses = GuestHouse::orderBy('name')->get();

        return view('expenses.edit', compact('expense', 'beneficiaries', 'projects', 'campaigns', 'workspaces', 'guestHouses'));
    }

    public function update(Request $request, Expense $expense)
    {
        $data = $request->validate([
            'type' => 'sometimes|in:operational,aid,logistics',
            'category' => 'nullable|string|max:100',
            'amount' => 'nullable|numeric',
            'currency' => 'nullable|string',
            'payment_method' => 'nullable|string',
            'description' => 'nullable|string',
            'project_id' => 'nullable|exists:projects,id',
            'campaign_id' => 'nullable|exists:campaigns,id',
            'workspace_id' => 'nullable|exists:workspaces,id',
            'guest_house_id' => 'nullable|exists:guest_houses,id',
            'beneficiary_id' => 'nullable|exists:beneficiaries,id',
            'paid_at' => 'nullable|date'
        ]);
        $expense->update($data);
        return redirect()->route('expenses.show', $expense);
    }

    public function destroy(Expense $expense)
    {
        $expense->delete();
        return redirect()->route('expenses.index');
    }
}
