<?php
namespace App\Http\Controllers;

use App\Models\Account;
use Illuminate\Http\Request;

class AccountWebController extends Controller
{
    public function index()
    {
        $accounts = Account::whereNull('parent_id')->with('children')->orderBy('type')->orderBy('code')->get();
        $allAccounts = Account::with('parent')->orderBy('type')->orderBy('code')->paginate(50);
        return view('accounts.index', compact('accounts', 'allAccounts'));
    }

    public function create()
    {
        $parents = Account::orderBy('code')->get();
        return view('accounts.create', compact('parents'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'code' => 'required|string|unique:accounts,code',
            'name' => 'required|string',
            'type' => 'required|in:asset,liability,equity,revenue,expense',
            'parent_id' => 'nullable|exists:accounts,id',
            'description' => 'nullable|string'
        ]);
        Account::create($data);
        return redirect()->route('accounts.index');
    }

    public function show(Account $account)
    {
        $account->load('parent', 'children');
        $lines = $account->lines()->with('journalEntry')->orderByDesc('id')->paginate(20);
        return view('accounts.show', compact('account', 'lines'));
    }

    public function edit(Account $account)
    {
        $parents = Account::where('id', '<>', $account->id)->orderBy('code')->get();
        return view('accounts.edit', compact('account', 'parents'));
    }

    public function update(Request $request, Account $account)
    {
        $data = $request->validate([
            'code' => 'required|string|unique:accounts,code,' . $account->id,
            'name' => 'required|string',
            'type' => 'required|in:asset,liability,equity,revenue,expense',
            'parent_id' => 'nullable|exists:accounts,id',
            'description' => 'nullable|string'
        ]);
        $account->update($data);
        return redirect()->route('accounts.index');
    }

    public function destroy(Account $account)
    {
        if ($account->children()->exists() || $account->lines()->exists()) {
            return back()->withErrors(['error' => 'لا يمكن حذف حساب لديه أبناء أو قيود']);
        }
        $account->delete();
        return redirect()->route('accounts.index');
    }
}
